#!/bin/bash

# © ООО «1С-Софт», 1996-2025. Все права защищены
# Copyright © 1C-Soft LLC, 1996-2025. All rights reserved

if [ `which lsb_release 2>/dev/null` ]; then
    distrName=`lsb_release -is`
else
    distrName=`cat /etc/os-release | grep ^ID= | cut -d= -f2 | tr -d '"'`
fi

# if system has dnf, prefer it to yum
[ `which dnf 2>/dev/null` ] && YUM=dnf || YUM=yum

function failedPacks_format()
{
    [ -z "$failedPacks" ] && failedPacks="$1" || failedPacks="$failedPacks $1"
}

function check_dpkg()
{
    [ -z "$APTPACKS" ] && { APTPACKS=`dpkg -l | awk '/^[hi]i/{print $2}'`; }
    test `echo "$APTPACKS" | grep "$1" | wc -l` -eq 0
}

function install_apt()
{
    # ALT uses "apt-get" but "rpm"
    if which dpkg >/dev/null 2>&1; then
        if check_dpkg "$1"; then
            apt-get install -y $1 >/dev/null 2>&1 || failedPacks_format $1
        fi
    else
        if check_rpm "$1"; then
            apt-get install -y $1 >/dev/null 2>&1 || failedPacks_format $1
        fi
    fi
}

function install_apt_final_error_msg()
{
    [ -z "$failedPacks" ] || ERRORMSG=`apt-get install -y "$failedPacks" 2>&1>/dev/null`
}

function install_dpkg()
{
    if check_dpkg "$1"; then
        dpkg -i $1 >/dev/null 2>&1 || failedPacks_format $1
    fi
}

function check_rpm()
{
    [ -z "$YUMPACKS" ] && { YUMPACKS=`rpm -qa --queryformat '%{NAME}\n'`; }
    test `echo "$YUMPACKS" | grep -Fx "$1" | wc -l` -eq 0
}

function install_yum()
{
    if check_rpm "$1"; then
        $YUM install -y $1 >/dev/null 2>&1 || failedPacks_format $1
    fi
}

function install_yum_final_error_msg()
{
    [ -z "$failedPacks" ] || ERRORMSG=`$YUM install -y "$failedPacks" 2>&1>/dev/null`
}

function install_custom_mscore_fonts()
{
    [ -z "$1" ] && FONTDIR="/usr/share/fonts/truetype/msttcorefonts" || FONTDIR="$1"
    [ `find $FONTDIR -name "*ndale*.ttf" -o -name "*ri*.ttf" -o -name "*omic*.ttf" -o -name "*our*.ttf" -o -name "*eorg*.ttf" -o -name "*mpact*.ttf" -o -name "*imes*.ttf" -o -name "*rebuc*.ttf" -o -name "*erdana*.ttf" -o -name "*ebding*.ttf" 2>/dev/null | wc -l` -ge 60 ] && exit 0

    mkdir -p $FONTDIR

    URLROOTS="http://downloads.sourceforge.net/corefonts/
            http://switch.dl.sourceforge.net/sourceforge/corefonts/
            http://dfn.dl.sourceforge.net/sourceforge/corefonts/
            http://heanet.dl.sourceforge.net/sourceforge/corefonts/
            http://jaist.dl.sourceforge.net/sourceforge/corefonts/
            http://nchc.dl.sourceforge.net/sourceforge/corefonts/
            http://ufpr.dl.sourceforge.net/sourceforge/corefonts/
            http://internode.dl.sourceforge.net/sourceforge/corefonts/
            http://voxel.dl.sourceforge.net/sourceforge/corefonts/
            http://kent.dl.sourceforge.net/sourceforge/corefonts/"

    SCRATCHDIR=`mktemp -t -d msttcorefonts.XXXXXX`
    chmod 0755 $SCRATCHDIR
    cd $SCRATCHDIR

cat <<EOF > msfonts.info
48d9bc613917709d3b0e0f4a6d4fe33a5c544c5035dffe9e90bc11e50e822071	Andale_Mono.ttf		andale32.exe	andalemo.ttf
dad7c04acb26e23dfe4780e79375ca193ddaf68409317e81577a30674668830e	Arial_Black.ttf		arialb32.exe	ariblk.ttf
35c0f3559d8db569e36c31095b8a60d441643d95f59139de40e23fada819b833	Arial.ttf		arial32.exe	arial.ttf
4044aa6b5bebbc36980206b45b0aaaaa5681552a48bcadb41746d5d1d71fd7b4	Arial_Bold.ttf		arial32.exe	arialbd.ttf
2f371cd9d96b3ac544519d85c16dc43ceacdfcea35090ee8ddf3ec5857c50328	Arial_Bold_Italic.ttf	arial32.exe	arialbi.ttf
70ade233175a6a6675e4501461af9326e6f78b1ffdf787ca0da5ab0fc8c9cfd6	Arial_Italic.ttf	arial32.exe	ariali.ttf
b82c53776058f291382ff7e008d4675839d2dc21eb295c66391f6fb0655d8fc0	Comic_Sans_MS.ttf	comic32.exe	comic.ttf
873361465d994994762d0b9845c99fc7baa2a600442ea8db713a7dd19f8b0172	Comic_Sans_MS_Bold.ttf	comic32.exe	comicbd.ttf
6715838c52f813f3821549d3f645db9a768bd6f3a43d8f85a89cb6875a546c61	Courier_New.ttf		courie32.exe	cour.ttf
edf8a7c5bfcac2e1fe507faab417137cbddc9071637ef4648238d0768c921e02	Courier_New_Bold.ttf	courie32.exe	courbd.ttf
f3f6b09855b6700977e214aab5eb9e5be6813976a24f894bd7766e92c732fbe1	Courier_New_Italic.ttf	courie32.exe	couri.ttf
66dbfa20b534fba0e203da140fec7276a45a1069e424b1b9c35547538128bbe8	Courier_New_Bold_Italic.ttf	courie32.exe	courbi.ttf
7d0bb20c632bb59e81a0885f573bd2173f71f73204de9058feb68ce032227072	Georgia.ttf		georgi32.exe	georgia.ttf
82d2fbadb88a8632d7f2e8ad50420c9fd2e7d3cbc0e90b04890213a711b34b93	Georgia_Bold.ttf	georgi32.exe	georgiab.ttf
1523f19bda6acca42c47c50da719a12dd34f85cc2606e6a5af15a7728b377b60	Georgia_Italic.ttf	georgi32.exe	georgiai.ttf
c983e037d8e4e694dd0fb0ba2e625bca317d67a41da2dc81e46a374e53d0ec8a	Georgia_Bold_Italic.ttf	georgi32.exe	georgiaz.ttf
00f1fc230ac99f9b97ba1a7c214eb5b909a78660cb3826fca7d64c3af5a14848	Impact.ttf		impact32.exe	impact.ttf
4e98adeff8ccc8ef4e3ece8d4547e288ff85fdc9c7ca711a4599c234874bbe86	Times_New_Roman.ttf	times32.exe	times.ttf
4357b63cef20c01661a53c5dae70ffd20cb4765503aaed6d38b17a57c5a90bff	Times_New_Roman_Bold.ttf	times32.exe	timesbd.ttf
192e1b0d18e90334e999a99f8c32808d6a2e74b3698b8cd90c943c2249a46549	Times_New_Roman_Bold_Italic.ttf	times32.exe	timesbi.ttf
c25ae529b4cecdbca148b6ccb862ee0abad770af8b1fd29c8dba619d1b8da78a	Times_New_Roman_Italic.ttf	times32.exe	timesi.ttf
ec3ffb302488251e1b67fb09dd578b364c5339e27c1cfb26eb627666236453d0	Trebuchet_MS.ttf	trebuc32.exe	trebuc.ttf
f65941f9487c0a0a3b7445996ecbbd24466d7ae76ea2a597ced55f438fa63838	Trebuchet_MS_Bold.ttf	trebuc32.exe	trebucbd.ttf
db56fdac7d3ba20b7aededcb6ee86c46687489d17b759e1708ea4e2d21e38410	Trebuchet_MS_Italic.ttf	trebuc32.exe	trebucit.ttf
c0a6bdf31f9f2953b2f08a0c1734c892bc825f0fb17c604d420f7acf203a213b	Trebuchet_MS_Bold_Italic.ttf	trebuc32.exe	trebucbi.ttf
96ed14949ca4b7392cff235b9c41d55c125382abbe0c0d3c2b9dd66897cae0cb	Verdana.ttf		verdan32.exe	verdana.ttf
c8f5065ba91680f596af3b0378e2c3e713b95a523be3d56ae185ca2b8f5f0b23	Verdana_Bold.ttf	verdan32.exe	verdanab.ttf
91b59186656f52972531a11433c866fd56e62ec4e61e2621a2dba70c8f19a828	Verdana_Italic.ttf	verdan32.exe	verdanai.ttf
698e220f48f4a40e77af7eb34958c8fd02f1e18c3ba3f365d93bfa2ed4474c80	Verdana_Bold_Italic.ttf	verdan32.exe	verdanaz.ttf
10d099c88521b1b9e380b7690cbe47b54bb19396ca515358cfdc15ac249e2f5d	Webdings.ttf		webdin32.exe	webdings.ttf
EOF

    for ttf in `awk '{print $2}' msfonts.info` ; do
        if [ ! -e $FONTDIR/$ttf ] || \
            [ `sha256sum $FONTDIR/$ttf | awk '{print $1}'` != `awk "/$ttf/ {print \\$1 }" msfonts.info` ]
            then
            THISFILE=`grep $ttf msfonts.info | awk '{print $3}'`
            if ! echo $FONTFILES | grep -q $THISFILE ; then
                FONTFILES="$FONTFILES $THISFILE"
            fi
        fi
    done
    for ff in $FONTFILES; do
        for URLROOT in $URLROOTS ; do
            if [ ! -e $ff.done ] || [ ! -e $ff ] ; then
                if ! wget --continue --tries=1 --connect-timeout=30 --read-timeout=150 --quiet --directory-prefix . --no-directories --no-background --progress=dot:default $URLROOT$ff ; then
                    continue 1
                fi
                touch $ff.done
                break
            fi
        done
        [ -e "$ff" ] || exit 1
    done
    for ff in $FONTFILES; do
        cabextract -q $ff 1>&2
        rm $ff
    done
    for x in *; do
        y=`echo $x | tr '[A-Z]' '[a-z]'`
        if [ "$x" != "$y" ]; then
            mv "$x" "$y"
        fi
    done
    chmod 644 *
    for ff in $FONTFILES; do
        for ttf in `grep $ff msfonts.info | awk '{print $4}'`; do
            longname=`awk "/$ttf/ { print \\$2 }" msfonts.info`
            if [ ! -e $FONTDIR/$longname ]; then
                mv $ttf $FONTDIR/$longname
                ln -sf $longname $FONTDIR/$ttf
                needFCcache=1
            fi
        done
    done
    cd /
    rm -rf $SCRATCHDIR
}

function install_dpkg_ttfmscorefontsinstaller()
{
    if check_dpkg "ttf-mscorefonts-installer"; then
        for p in debconf xfonts-utils cabextract fontconfig; do install_apt $p; done
        [ -z "$failedPacks" ] || return 1;
        echo ttf-mscorefonts-installer msttcorefonts/accepted-mscorefonts-eula select true | debconf-set-selections >/dev/null 2>&1
        install_apt "ttf-mscorefonts-installer"
        [ -z "$failedPacks" ] || { install_custom_mscore_fonts "/usr/share/fonts/truetype/msttcorefonts" && { fc-cache -f >/dev/null 2>&1; unset failedPacks; }; }
    fi
}

function install_yum_msttcorefontsinstaller()
{
    if check_rpm "msttcore-fonts-installer"; then
        for p in curl cabextract xorg-x11-font-utils fontconfig; do install_yum $p; done
        $YUM install -y https://downloads.sourceforge.net/project/mscorefonts2/rpms/msttcore-fonts-installer-2.6-1.noarch.rpm || failedPacks_format "msttcore-fonts-installer"
        [ -z "$failedPacks" ] || { install_custom_mscore_fonts "/usr/share/fonts/msttcore" && fc-cache -f >/dev/null 2>&1; }
    fi
}

# Installation modes
for mode in "$@"; do
    [[ "$mode" == common* ]] && commonMode="yes"
    [[ "$mode" == ui* ]] && uiMode="yes"
done
# nothing specific - install everything
[[ -z "$commonMode" && -z "$uiMode" ]] && allMode="yes" || :

if [ ! -z "$commonMode" ] || [ ! -z "$allMode" ]; then
    case "$distrName" in
        "ALT"*| \
        "altlinux"*)
            # Here apt-get but RPM check!
            apt-get install -y libfreetype libcairo libharfbuzz libfribidi libgsf liblcms2 libkrb5 libcrypto[0-9] >/dev/null 2>&1
            if check_rpm "fonts-ttf-ms"; then
                for p in cabextract fontconfig; do install_apt $p; done
                # Если придется принимать лицензию EULA, то процесс установки может зависнуть
                # install_apt "fonts-ttf-ms"
                # rpm -i https://mirror.yandex.ru/altlinux/p9/branch/files/SRPMS/fonts-ttf-ms-1.0-alt4.src.rpm || failedPacks_format "fonts-ttf-ms"
                [ -z "$failedPacks" ] || install_custom_mscore_fonts "/usr/share/fonts/ttf/ms"
            fi
            # NO packages debconf xfonts-utils. EXISTS cabextract
            # FONTDIR=/usr/share/fonts/ttf/ms
            # install_apt "fonts-ttf-ms"
        ;;
        "RED SOFT"*| \
        "redos"*| \
        "RedHatEnterprise"*| \
        "rhel"*| \
        "CentOS"*| \
        "centos"*)
            $YUM install -y freetype cairo harfbuzz fribidi lcms2 krb5-libs >/dev/null 2>&1
            install_yum_msttcorefontsinstaller
        ;;
        "AstraLinux"*| \
        "astra"*| \
        "Debian"*| \
        "debian"*| \
        "LinuxMint"*| \
        "linuxmint"*| \
        "Ubuntu"*| \
        "ubuntu"*| \
        *)
            apt-get install -y libfreetype[0-9] libcairo[0-9] libharfbuzz[0-9] libfribidi[0-9] libgsf-bin liblcms2-[0-9] libkrb5-[0-9] libgssapi-krb5-[0-9] >/dev/null 2>&1
            install_dpkg_ttfmscorefontsinstaller
        ;;
    esac
fi

if [ ! -z "$uiMode" ] || [ ! -z "$allMode" ]; then
    case "$distrName" in
        "ALT"*| \
        "altlinux"*)
            for p in polkit zenity xmessage evince libwebkit2gtk libGLU; do install_apt $p; done
            install_apt_final_error_msg
        ;;
        "Fedora"*| \
        "fedora"*)
            for p in polkit zenity xmessage evince webkit2gtk3 mesa-libGLU; do install_yum $p; done
            install_yum_final_error_msg
        ;;
        "OracleServer"*| \
        "ol"*| \
        "RedHatEnterprise"*| \
        "rhel"*)
            for p in polkit zenity evince webkit2gtk3 mesa-libGLU; do install_yum $p; done
            install_yum_final_error_msg
        ;;
        "RED SOFT"*| \
        "redos"*)
            for p in polkit zenity xorg-x11-apps evince webkit2gtk3 mesa-libGLU; do install_yum $p; done
            install_yum_final_error_msg
        ;;
        "CentOS"*| \
        "centos"*)
            for p in polkit zenity xorg-x11-apps evince mesa-libGLU; do install_yum $p; done
            if [ `which lsb_release 2>/dev/null` ]; then
                distrVer=`lsb_release -r | awk '{print $NF}' | cut -d'.' -f1`
            else
                distrVer=`cat /etc/os-release | grep "^VERSION_ID=" | cut -d'=' -f2 | cut -d'.' -f1 | tr -d '"'`
            fi
            [ "$distrVer" -le "7" ] && install_yum webkitgtk4 || install_yum webkit2gtk3
            install_yum_final_error_msg
         ;;
         "Astra"*| \
         "astra"*)
            for p in policykit-1 x11-utils okular libglu1-mesa libwebkit2gtk-4.[0-9]-[0-9]; do install_apt $p; done
            install_apt_final_error_msg
         ;;
        *)
            for p in policykit-1 zenity x11-utils evince libglu1-mesa libwebkit2gtk-4.[0-9]-[0-9]; do install_apt $p; done
            install_apt_final_error_msg
        ;;
    esac
    [ -z "$failedPacks" ] || { NL=$'\n'; echo "$NL$failedPacks$NL$NL$ERRORMSG"; }
fi
